#include <string>
#include <vector>
#include <stdio.h>
#include <stdlib.h>

#define TXMP_STRING_TYPE	std::string

#include "XMP.hpp"
#include "XMP.incl_cpp"

#include "MD5.h"
typedef XMP_Uns8 MD5_Digest[16];	// ! Should be in MD5.h.

using namespace std;

#if XMP_WinBuild
	#pragma warning ( disable : 4244 )	// integer shortening, possible loss of data
	#pragma warning ( disable : 4267 )	// integer shortening, possible loss of data
	#pragma warning ( disable : 4996 )	// '...' was declared deprecated
#endif

#pragma pack ( 1 )

// =================================================================================================

FILE * sLogFile = 0;

long kOne = 1;
char firstByte = *((char*)&kOne);

const bool sBigEndianHost = (firstByte == 0);
const bool sLittleEndianHost = (firstByte == 1);

static bool beTIFF;

typedef const char * ChPtr;
#define CheckBytes(left,right,len)	( memcmp ( ((ChPtr)(left)), ((ChPtr)(right)), len ) == 0 )

static XMP_Uns8* sDataPtr = 0;	// Used via CaptureFileData for variable length data.
static XMP_Uns32 sDataMax = 0;
static XMP_Uns32 sDataLen = 0;

static XMP_Uns8* sXMPPtr = 0;	// Used via CaptureXMP for the main XMP.
static XMP_Uns32 sXMPMax = 0;
static XMP_Uns32 sXMPLen = 0;
static XMP_Int64 sXMPPos = 0;

static bool sHaveOldDigest = false;
static MD5_Digest sOldDigest;

typedef XMP_Uns16 (*GetUns16_Proc) ( const void * addr );
typedef XMP_Uns32 (*GetUns32_Proc) ( const void * addr );
typedef XMP_Uns64 (*GetUns64_Proc) ( const void * addr );

static XMP_Uns16 GetUns16BE ( const void * addr );
static XMP_Uns16 GetUns16LE ( const void * addr );
static XMP_Uns32 GetUns32BE ( const void * addr );
static XMP_Uns32 GetUns32LE ( const void * addr );
static XMP_Uns64 GetUns64BE ( const void * addr );
static XMP_Uns64 GetUns64LE ( const void * addr );

#define High32(u64) ((XMP_Uns32)((u64) >> 32))
#define Low32(u64)  ((XMP_Uns32)((u64) & 0xFFFFFFFFUL))

// =================================================================================================

static void DumpTIFF ( XMP_Uns8 * tiffContent, XMP_Uns32 tiffLen, XMP_Uns32 fileOffset, const char * label );
static void DumpIPTC ( XMP_Uns8 * iptcOrigin, XMP_Uns32 iptcLen, XMP_Uns32 fileOffset, const char * label );
static void DumpImageResources ( XMP_Uns8 * psirOrigin, XMP_Uns32 psirLen, XMP_Uns32 fileOffset, const char * label );
static void DumpIFDChain ( XMP_Uns8 * startPtr, XMP_Uns8 * endPtr, XMP_Uns8 * tiffContent, XMP_Uns32 fileOffset, const char * label );

static GetUns16_Proc TIFF_GetUns16 = 0;	// Keeps endian procs for the "current" TIFF.
static GetUns32_Proc TIFF_GetUns32 = 0;
static GetUns64_Proc TIFF_GetUns64 = 0;

enum {	// Special TIFF tags
	kTIFF_XMP       = 700,
	kTIFF_IPTC      = 33723,
	kTIFF_PSIR      = 34377,
	kTIFF_Exif      = 34665,
	kTIFF_GPS       = 34853,
	kTIFF_MakerNote = 37500,
	kTIFF_Interop   = 40965
};

enum {	// Special Photoshop image resource IDs
	kPSIR_OldCaption	= 1008,
	kPSIR_PrintCaption	= 1020,
	kPSIR_IPTC			= 1028,
	kPSIR_CopyrightFlag	= 1034,
	kPSIR_CopyrightURL	= 1035,
	kPSIR_Exif_1		= 1058,
	kPSIR_Exif_3		= 1059,
	kPSIR_XMP			= 1060,
	kPSIR_IPTC_Digest	= 1061
};

struct IPTC_DataSet {	// The 5 byte header of an IPTC DataSet.
	XMP_Uns8 tagMarker;
	XMP_Uns8 recordNumber;
	XMP_Uns8 dataSetNumber;
	XMP_Uns8 octetCountHigh;
	XMP_Uns8 octetCountLow;
};

enum {	// IPTC DataSet IDs
	kIPTC_IntellectualGenre =   4,
	kIPTC_Title             =   5,
	kIPTC_Urgency           =  10,
	kIPTC_SubjectCode       =  12,
	kIPTC_Category          =  15,
	kIPTC_SuppCategory      =  20,
	kIPTC_Keyword           =  25,
	kIPTC_Instructions      =  40,
	kIPTC_DateCreated       =  55,
	kIPTC_TimeCreated       =  60,
	kIPTC_DigitalCreateDate =  62,
	kIPTC_DigitalCreateTime =  63,
	kIPTC_Creator           =  80,
	kIPTC_CreatorJobtitle   =  85,
	kIPTC_City              =  90,
	kIPTC_Location          =  92,
	kIPTC_State             =  95,
	kIPTC_CountryCode       = 100,
	kIPTC_Country           = 101,
	kIPTC_JobID             = 103,
	kIPTC_Headline          = 105,
	kIPTC_Provider          = 110,
	kIPTC_Source            = 115,
	kIPTC_CopyrightNotice   = 116,
	kIPTC_Description       = 120,
	kIPTC_DescriptionWriter = 122 
};

struct DataSetInfo {
	XMP_Uns8 id;
	const char * name;
};

static const DataSetInfo kDataSetNames[] =
	{ { kIPTC_IntellectualGenre, "Intellectual Genre" },
	  { kIPTC_Title, "Title" },
	  { kIPTC_Urgency, "Urgency" },
	  { kIPTC_SubjectCode, "Subject Code" },
	  { kIPTC_Category, "Category" },
	  { kIPTC_SuppCategory, "Supplemental Category" },
	  { kIPTC_Keyword, "Keyword" },
	  { kIPTC_Instructions, "Instructions" },
	  { kIPTC_DateCreated, "Date Created" },
	  { kIPTC_TimeCreated, "Time Created" },
	  { kIPTC_DigitalCreateDate, "Digital Date Created" },
	  { kIPTC_DigitalCreateTime, "Digital Time Created" },
	  { kIPTC_Creator, "Creator" },
	  { kIPTC_CreatorJobtitle, "Creator Jobtitle" },
	  { kIPTC_City, "City" },
	  { kIPTC_Location, "Location" },
	  { kIPTC_State, "Province-State" },
	  { kIPTC_CountryCode, "Country Code" },
	  { kIPTC_Country, "Country" },
	  { kIPTC_JobID, "Job ID" },
	  { kIPTC_Headline, "Headline" },
	  { kIPTC_Provider, "Provider" },
	  { kIPTC_Source, "Source" },
	  { kIPTC_CopyrightNotice, "Copyright Notice" },
	  { kIPTC_Description, "Description" },
	  { kIPTC_DescriptionWriter, "Description Writer" },
	  { 0, 0 } };

enum {
	kTIFF_Uns8		=  1,
	kTIFF_ASCII		=  2,
	kTIFF_Uns16		=  3,
	kTIFF_Uns32		=  4,
	kTIFF_URational	=  5,
	kTIFF_Int8		=  6,
	kTIFF_Undef8	=  7,
	kTIFF_Int16		=  8,
	kTIFF_Int32		=  9,
	kTIFF_SRational	= 10,
	kTIFF_Float		= 11,
	kTIFF_Double	= 12,
	kTIFF_TypeEnd
};

static const int sTIFF_TypeSizes[] = { 0, 1, 1, 2, 4, 8, 1, 1, 2, 4, 8, 4, 8 };
static const char * sTIFF_TypeNames[] = { "", "BYTE", "ASCII", "SHORT", "LONG", "RATIONAL",
										  "SBYTE", "UNDEFINED", "SSHORT", "SLONG", "SRATIONAL",
										  "FLOAT", "DOUBLE" };

struct TagNameInfo {
	long	tag;
	const char * name;
};

static const TagNameInfo sTIFF_TagNames[] =
	{ { 254, "NewSubfileType" },
	  { 255, "SubfileType" },
	  { 256, "ImageWidth" },
	  { 257, "ImageLength" },
	  { 258, "BitsPerSample" },
	  { 259, "Compression" },
	  { 262, "PhotometricInterpretation" },
	  { 263, "Thresholding" },
	  { 264, "CellWidth" },
	  { 265, "CellLength" },
	  { 266, "FillOrder" },
	  { 269, "DocumentName" },
	  { 270, "ImageDescription" },
	  { 271, "Make" },
	  { 272, "Model" },
	  { 273, "StripOffsets" },
	  { 274, "Orientation" },
	  { 277, "SamplesPerPixel" },
	  { 278, "RowsPerStrip" },
	  { 279, "StripByteCounts" },
	  { 280, "MinSampleValue" },
	  { 281, "MaxSampleValue" },
	  { 282, "XResolution" },
	  { 283, "YResolution" },
	  { 284, "PlanarConfiguration" },
	  { 285, "PageName" },
	  { 286, "XPosition" },
	  { 287, "YPosition" },
	  { 288, "FreeOffsets" },
	  { 289, "FreeByteCounts" },
	  { 290, "GrayResponseUnit" },
	  { 291, "GrayResponseCurve" },
	  { 292, "T4Options" },
	  { 293, "T6Options" },
	  { 296, "ResolutionUnit" },
	  { 297, "PageNumber" },
	  { 301, "TransferFunction" },
	  { 305, "Software" },
	  { 306, "DateTime" },
	  { 315, "Artist" },
	  { 316, "HostComputer" },
	  { 317, "Predictor" },
	  { 318, "WhitePoint" },
	  { 319, "PrimaryChromaticities" },
	  { 320, "ColorMap" },
	  { 321, "HalftoneHints" },
	  { 322, "TileWidth" },
	  { 323, "TileLength" },
	  { 324, "TileOffsets" },
	  { 325, "TileByteCounts" },
	  { 330, "SubIFDs" },
	  { 332, "InkSet" },
	  { 333, "InkNames" },
	  { 334, "NumberOfInks" },
	  { 336, "DotRange" },
	  { 337, "TargetPrinter" },
	  { 338, "ExtraSamples" },
	  { 339, "SampleFormat" },
	  { 340, "SMinSampleValue" },
	  { 341, "SMaxSampleValue" },
	  { 342, "TransferRange" },
	  { 347, "JPEGTables" },
	  { 512, "JPEGProc" },
	  { 513, "JPEGInterchangeFormat" },
	  { 514, "JPEGInterchangeFormatLength" },
	  { 515, "JPEGRestartInterval" },
	  { 517, "JPEGLosslessPredictors" },
	  { 518, "JPEGPointTransforms" },
	  { 519, "JPEGQTables" },
	  { 520, "JPEGDCTables" },
	  { 521, "JPEGACTables" },
	  { 529, "YCbCrCoefficients" },
	  { 530, "YCbCrSubSampling" },
	  { 531, "YCbCrPositioning" },
	  { 532, "ReferenceBlackWhite" },
	  { 33421, "CFARepeatPatternDim" },
	  { 33422, "CFAPattern" },
	  { 33423, "BatteryLevel" },
	  { 33432, "Copyright" },
	  { 33434, "ExposureTime" },
	  { 33437, "FNumber" },
	  { 33723, "IPTC/NAA" },
	  { 34675, "InterColourProfile" },
	  { 34850, "ExposureProgram" },
	  { 34852, "SpectralSensitivity" },
	  { 34853, "GPSInfo" },
	  { 34855, "ISOSpeedRatings" },
	  { 34856, "OECF" },
	  { 34857, "Interlace" },
	  { 34858, "TimeZoneOffset" },
	  { 34859, "SelfTimerMode" },
	  { 36864, "ExifVersion" },
	  { 36867, "DateTimeOriginal" },
	  { 36868, "DateTimeDigitized" },
	  { 37121, "ComponentsConfiguration" },
	  { 37122, "CompressedBitsPerPixel" },
	  { 37377, "ShutterSpeedValue" },
	  { 37378, "ApertureValue" },
	  { 37379, "BrightnessValue" },
	  { 37380, "ExposureBiasValue" },
	  { 37381, "MaxApertureValue" },
	  { 37382, "SubjectDistance" },
	  { 37383, "MeteringMode" },
	  { 37384, "LightSource" },
	  { 37385, "Flash" },
	  { 37386, "FocalLength" },
	  { 37387, "FlashEnergy" },
	  { 37388, "SpatialFrequencyResponse" },
	  { 37389, "Noise" },
	  { 37390, "FocalPlaneXResolution" },
	  { 37391, "FocalPlaneYResolution" },
	  { 37392, "FocalPlaneResolutionUnit" },
	  { 37393, "ImageNumber" },
	  { 37394, "SecurityClassification" },
	  { 37395, "ImageHistory" },
	  { 37396, "SubjectLocation" },
	  { 37397, "ExposureIndex" },
	  { 37398, "TIFF/EPStandardID" },
	  { 37399, "SensingMethod" },
	  { 37396, "SubjectArea" },
	  { 37500, "MakerNote" },
	  { 37510, "UserComment" },
	  { 37520, "SubSecTime" },
	  { 37521, "SubSecTimeOriginal" },
	  { 37522, "SubSecTimeDigitized" },
	  { 37724, "ImageSourceData " },
	  { 40960, "FlashpixVersion" },
	  { 40961, "ColorSpace" },
	  { 40962, "PixelXDimension" },
	  { 40963, "PixelYDimension" },
	  { 40964, "RelatedSoundFile" },
	  { 41483, "FlashEnergy" },
	  { 41484, "SpatialFrequencyResponse" },
	  { 41486, "FocalPlaneXResolution" },
	  { 41487, "FocalPlaneYResolution" },
	  { 41488, "FocalPlaneResolutionUnit" },
	  { 41492, "SubjectLocation" },
	  { 41493, "ExposureIndex" },
	  { 41495, "SensingMethod" },
	  { 41728, "FileSource" },
	  { 41729, "SceneType" },
	  { 41730, "CFAPattern" },
	  { 41985, "CustomRendered" },
	  { 41986, "ExposureMode" },
	  { 41987, "WhiteBalance" },
	  { 41988, "DigitalZoomRatio" },
	  { 41989, "FocalLengthIn35mmFilm" },
	  { 41990, "SceneCaptureType" },
	  { 41991, "GainControl" },
	  { 41992, "Contrast" },
	  { 41993, "Saturation" },
	  { 41994, "Sharpness" },
	  { 41995, "DeviceSettingDescription" },
	  { 41996, "SubjectDistanceRange" },
	  { 42016, "ImageUniqueID" },
	  { 50706, "DNGVersion" },
	  { 50707, "DNGBackwardVersion" },
	  { 50708, "DNG UniqueCameraModel" },
	  { 0, "" } };

// =================================================================================================

static XMP_Uns16
GetUns16BE ( const void * addr )	// Get a 16 bit integer that is in memory as big endian.
{

	if ( sBigEndianHost ) {
		XMP_Uns16 * ptr = (XMP_Uns16*)addr;
		return *ptr;
	} else {
		XMP_Uns8 * ptr = (XMP_Uns8*)addr;
		XMP_Uns16 result = ptr[0];
		result = (result << 8) + ptr[1];
		return result;
	}

}

// -------------------------------------------------------------------------------------------------

static XMP_Uns16
GetUns16LE ( const void * addr )	// Get a 16 bit integer that is in memory as little endian.
{

	if ( sLittleEndianHost ) {
		XMP_Uns16 * ptr = (XMP_Uns16*)addr;
		return *ptr;
	} else {
		XMP_Uns8 * ptr = (XMP_Uns8*)addr;
		XMP_Uns16 result = ptr[1];
		result = (result << 8) + ptr[0];
		return result;
	}

}

// -------------------------------------------------------------------------------------------------

static XMP_Uns32
GetUns32BE ( const void * addr )	// Get a 32 bit integer that is in memory as big endian.
{

	if ( sBigEndianHost ) {
		XMP_Uns32 * ptr = (XMP_Uns32*)addr;
		return *ptr;
	} else {
		XMP_Uns8 * ptr = (XMP_Uns8*)addr;
		XMP_Uns32 result = ptr[0];
		for ( int i = 1; i <= 3; ++i ) result = (result << 8) + ptr[i];
		return result;
	}

}

// -------------------------------------------------------------------------------------------------

static XMP_Uns32
GetUns32LE ( const void * addr )	// Get a 32 bit integer that is in memory as little endian.
{

	if ( sLittleEndianHost ) {
		XMP_Uns32 * ptr = (XMP_Uns32*)addr;
		return *ptr;
	} else {
		XMP_Uns8 * ptr = (XMP_Uns8*)addr;
		XMP_Uns32 result = ptr[3];
		for ( int i = 2; i >= 0; --i ) result = (result << 8) + ptr[i];
		return result;
	}

}

// -------------------------------------------------------------------------------------------------

static XMP_Uns64
GetUns64BE ( const void * addr )	// Get a 64 bit integer that is in memory as big endian.
{

	if ( sBigEndianHost ) {
		XMP_Uns64 * ptr = (XMP_Uns64*)addr;
		return *ptr;
	} else {
		XMP_Uns8 * ptr = (XMP_Uns8*)addr;
		XMP_Uns64 result = ptr[0];
		for ( int i = 1; i <= 7; ++i ) result = (result << 8) + ptr[i];
		return result;
	}

}

// -------------------------------------------------------------------------------------------------

static XMP_Uns64
GetUns64LE ( const void * addr )	// Get a 64 bit integer that is in memory as little endian.
{

	if ( sLittleEndianHost ) {
		XMP_Uns64 * ptr = (XMP_Uns64*)addr;
		return *ptr;
	} else {
		XMP_Uns8 * ptr = (XMP_Uns8*)addr;
		XMP_Uns64 result = ptr[7];
		for ( int i = 6; i >= 0; --i ) result = (result << 8) + ptr[i];
		return result;
	}

}

// =================================================================================================

static void
CaptureFileData ( FILE* file, XMP_Int64 offset, XMP_Uns32 length )
{

	if ( sDataPtr != 0 ) free ( sDataPtr );
	sDataPtr = (XMP_Uns8*) malloc ( length );
	fseek ( file, (long)offset, SEEK_SET );
	fread ( sDataPtr, 1, length, file );

}	// CaptureFileData

// -------------------------------------------------------------------------------------------------

static void
CaptureXMP ( const XMP_Uns8 * xmpPtr, const XMP_Uns32 xmpLen, XMP_Int64 fileOffset )
{

	if ( sXMPPtr != 0 ) free ( sXMPPtr );
	sXMPPtr = (XMP_Uns8*) malloc ( xmpLen );
	memcpy ( sXMPPtr, xmpPtr, xmpLen );

}	// CaptureXMP

// -------------------------------------------------------------------------------------------------

static void PrintOnlyASCII_8 ( XMP_Uns8 * strPtr, XMP_Uns32 strLen,
							   const char * label = 0, bool stopOnNUL = true, bool includeNewLine = true )
{
	bool prevBig = false;
	size_t remainder = 0;
	
	if ( (label != 0) && (*label != 0) ) printf ( "%s \"", label );
	
	for ( XMP_Uns32 i = 0; i < strLen; ++i ) {

		XMP_Uns8 ch = strPtr[i];
		
		if ( (0x20 <= ch) && (ch <= 0x7E) ) {
			if ( prevBig ) printf ( "\xA9" );
			printf ( "%c", ch );
			prevBig = false;
		} else {
			if ( ! prevBig ) printf ( "\xA9" );
			printf ( "%.2X", ch );
			prevBig = true;
			if ( (ch == 0) && stopOnNUL ) {
				remainder = strLen - (i + 1);
				break;
			}
		}
	
	}
	
	if ( prevBig ) printf ( "\xA9" );

	if ( (label != 0) && (*label != 0) ) {
		printf ( "\"" );
		if ( remainder != 0 ) printf ( ", ** remainder %d bytes **", remainder );
		if ( includeNewLine ) printf ( "\n" );
	}
	
}	// PrintOnlyASCII_8

// -------------------------------------------------------------------------------------------------

static void PrintOnlyASCII_16BE ( XMP_Uns16 * u16Ptr, XMP_Uns32 u16Bytes,
								  const char * label = 0, bool stopOnNUL = true, bool includeNewLine = true )
{
	bool prevBig = false;
	size_t remainder = 0;
	XMP_Uns32 u16Count = u16Bytes/2;
	
	if ( (label != 0) && (*label != 0) ) printf ( "%s \"", label );
	
	for ( XMP_Uns32 i = 0; i < u16Count; ++i ) {

		XMP_Uns16 u16 = u16Ptr[i];
		u16 = GetUns16BE ( &u16 );

		if ( (0x20 <= u16) && (u16 <= 0x7E) ) {
			if ( prevBig ) printf ( "\xA9" );
			printf ( "%c", u16 );
			prevBig = false;
		} else {
			if ( ! prevBig ) printf ( "\xA9" );
			printf ( "%.4X", u16 );
			prevBig = true;
			if ( (u16 == 0) && stopOnNUL ) {
				remainder = 2 * (u16Count - (i + 1));
				break;
			}
		}
	
	}
	
	if ( prevBig ) printf ( "\xA9" );

	if ( (label != 0) && (*label != 0) ) {
		printf ( "\"" );
		if ( remainder != 0 ) printf ( ", ** remainder %d bytes **", remainder );
		if ( includeNewLine ) printf ( "\n" );
	}
	
}	// PrintOnlyASCII_16BE

// -------------------------------------------------------------------------------------------------

static void PrintOnlyASCII_16LE ( XMP_Uns16 * u16Ptr, XMP_Uns32 u16Bytes,
								  const char * label = 0, bool stopOnNUL = true, bool includeNewLine = true )
{
	bool prevBig = false;
	size_t remainder = 0;
	XMP_Uns32 u16Count = u16Bytes/2;
	
	if ( (label != 0) && (*label != 0) ) printf ( "%s \"", label );
	
	for ( XMP_Uns32 i = 0; i < u16Count; ++i ) {

		XMP_Uns16 u16 = u16Ptr[i];
		u16 = GetUns16LE ( &u16 );

		if ( (0x20 <= u16) && (u16 <= 0x7E) ) {
			if ( prevBig ) printf ( "\xA9" );
			printf ( "%c", u16 );
			prevBig = false;
		} else {
			if ( ! prevBig ) printf ( "\xA9" );
			printf ( "%.4X", u16 );
			prevBig = true;
			if ( (u16 == 0) && stopOnNUL ) {
				remainder = 2 * (u16Count - (i + 1));
				break;
			}
		}
	
	}
	
	if ( prevBig ) printf ( "\xA9" );

	if ( (label != 0) && (*label != 0) ) {
		printf ( "\"" );
		if ( remainder != 0 ) printf ( ", ** remainder %d bytes **", remainder );
		if ( includeNewLine ) printf ( "\n" );
	}
	
}	// PrintOnlyASCII_16LE

// -------------------------------------------------------------------------------------------------

static XMP_Status
DumpCallback ( void * refCon, XMP_StringPtr outStr, XMP_StringLen outLen )
{
	bool & atStart = *((bool*)refCon);
	if ( (outLen == 1) && (*outStr == '\n') ) atStart = false;

	size_t nlCount = 0;
	while ( (outLen > 0) && (outStr[outLen-1] == '\n') ) {
		--outLen;
		++nlCount;
	}
	
	if ( ! atStart ) PrintOnlyASCII_8 ( (XMP_Uns8*)outStr, outLen );
	for ( ; nlCount > 0; --nlCount ) printf ( "\n" );

	return 0;

}	// DumpCallback

// -------------------------------------------------------------------------------------------------

static void
PrintXMPErrorInfo ( const XMP_Error & excep, const char * title )
{

	long id = excep.GetID();
	const char * message = excep.GetErrMsg();

	printf ( "%s\n", title );
	printf ( "   #%d : %s\n", id, message );

}	// PrintXMPErrorInfo

// =================================================================================================

static const size_t   kJPEGMinSize = 4;	// At least the SOI and EOI markers.
static const XMP_Uns8 kJPEGStart[] = { 0xFF, 0xD8, 0xFF };	// 0xFFD8 is SOI, plus 0xFF for next marker.

static const size_t   kTIFFMinSize = 8+2+12+4;	// At least the header plus 1 minimal IFD.
static const XMP_Uns8 kTIFFBigStart[]    = { 0x4D, 0x4D, 0x00, 0x2A };	// 0x4D is 'M', 0x2A is 42.
static const XMP_Uns8 kTIFFLittleStart[] = { 0x49, 0x49, 0x2A, 0x00 };	// 0x49 is 'I'.

static const size_t   kPhotoshopMinSize = 26 + 4*4;	// At least the file header and 4 section lengths.
static const XMP_Uns8 kPhotoshopV1Start[] = { 0x38, 0x42, 0x50, 0x53, 0x00, 0x01 };	// 0x38425053 is "8BPS".
static const XMP_Uns8 kPhotoshopV2Start[] = { 0x38, 0x42, 0x50, 0x53, 0x00, 0x02 };

static XMP_FileFormat
CheckFileFormat ( const char * filePath, XMP_Uns8 * fileContent, XMP_Uns32 fileSize )
{

	// ! The buffer passed to CheckFileFormat is just the first 4K bytes of the file.

	if ( (fileSize >= kJPEGMinSize) && CheckBytes ( fileContent, kJPEGStart, 3 ) ) {
		return kXMP_JPEGFile;
	}
	
	if ( (fileSize >= kTIFFMinSize) &&
		 (CheckBytes ( fileContent, kTIFFBigStart , 4 ) || CheckBytes ( fileContent, kTIFFLittleStart, 4 )) ) {
		return kXMP_TIFFFile;
	}
	
	if ( (fileSize >= kPhotoshopMinSize) &&
		 (CheckBytes ( fileContent, kPhotoshopV1Start, 6 ) || CheckBytes ( fileContent, kPhotoshopV2Start, 6 )) ) {
		return kXMP_PhotoshopFile;
	}
	
	return kXMP_UnknownFile;
	
}	// CheckFileFormat

// =================================================================================================
// DumpXMP
// =======

static void
DumpXMP ( XMP_Uns8 * xmpPtr, XMP_Uns32 xmpLen, XMP_Int64 xmpOffset, const char * label )
{
	if ( xmpOffset <= 0xFFFFFFFFUL ) {
		printf ( "XMP from %s, file offset 0x%X, size %d", label, (XMP_Uns32)xmpOffset, xmpLen );
	} else {
		printf ( "XMP from %s, file offset 0x%X-%.8X, size %d", label, High32(xmpOffset), Low32(xmpOffset), xmpLen );
	}

	bool atStart = true;
	SXMPMeta xmp ( (XMP_StringPtr)xmpPtr, xmpLen );
	xmp.DumpObject ( DumpCallback, &atStart );
	printf ( "\n" );

}	// DumpXMP

// =================================================================================================
// DumpXMP
// =======

static void
DumpXMP ( const char * label )
{
	DumpXMP ( sXMPPtr, sXMPLen, sXMPPos, label );

}	// DumpXMP

// =================================================================================================
// DumpIPTC
// ========
//
// The IPTC (IIM, NAA) values are in a sequence of "data sets". Each has a 5 byte header followed
// by the value. There is no overall length in the sequence itself. Photoshop writes this in TIFF
// as LONGs (4 byte chunks), so there might be padding at the end.

static void
DumpIPTC ( XMP_Uns8 * iptcOrigin, XMP_Uns32 iptcLen, XMP_Uns32 fileOffset, const char * label )
{
	printf ( "IPTC data from %s, file offset 0x%X, size %d\n", label, fileOffset, iptcLen );
	
	MD5_Digest newDigest;
	struct MD5_CTX md5Context;
	MD5Init ( &md5Context );
	MD5Update ( &md5Context, iptcOrigin, iptcLen );
	MD5Final ( newDigest, &md5Context );
	XMP_Uns32 * ndPtr = (XMP_Uns32*)(&newDigest);
	printf ( "   Digest = %.8X-%.8X-%.8X-%.8X",
			 GetUns32BE(ndPtr), GetUns32BE(ndPtr+1), GetUns32BE(ndPtr+2), GetUns32BE(ndPtr+3) );
	if ( ! sHaveOldDigest ) {
		printf ( "  (no old digest)\n\n" );
	} else if ( memcmp ( &newDigest, &sOldDigest, sizeof(MD5_Digest) ) == 0 ) {
		printf ( "  (old digest matches)\n\n" );
	} else {
		printf ( "  (*** old digest differs ***)\n\n" );
	}

	XMP_Uns8 * iptcPtr = iptcOrigin;
	XMP_Uns8 * iptcEnd = iptcPtr + iptcLen;
	XMP_Uns8 * valuePtr;
	XMP_Uns32 valueLen;

	while ( iptcPtr < (iptcEnd - 4) ) {	// ! The -4 is to skip terminal padding.

		IPTC_DataSet * currDS = (IPTC_DataSet*)iptcPtr;
		if ( currDS->tagMarker != 0x1C ) {
			printf ( "** invalid IPTC marker **\n" );
			break;
		}
		valuePtr = iptcPtr + 5;
		valueLen = (currDS->octetCountHigh << 8) + currDS->octetCountLow;

		if ( (valueLen >> 15) == 1 ) {
			int count = valueLen & 0x7FFF;
			valueLen = 0;
			for ( int i = 0; i < count; ++i ) valueLen = (valueLen << 8) + valuePtr[i];
			valuePtr += count;
		}
		
		XMP_Uns32 dsOffset = fileOffset + (iptcPtr - iptcOrigin);
		printf ( "   %d:%d, file offset 0x%X, size %d",
				  currDS->recordNumber, currDS->dataSetNumber, dsOffset, valueLen );
		
		if ( (currDS->recordNumber != 1) && (currDS->recordNumber != 2) ) {

			printf ( "\n" );

		} else if ( currDS->recordNumber == 1 ) {

			switch ( currDS->dataSetNumber ) {

				case 0 :
					{
					XMP_Uns16 version = GetUns16BE ( valuePtr );
					printf ( ", version = 0x%.4X\n", version );
					break;
					}
				case 90 :
					if ( valueLen == 3 ) {
						printf ( ", encoding = 0x%.2X%.2X%.2X", valuePtr[0], valuePtr[1], valuePtr[2] );
						if ( memcmp ( valuePtr, "\x1B\x25\x47", 3 ) == 0 ) printf ( " (UTF-8)" );
						}
					printf ( "\n" );
					break;
				default :
					printf ( "\n" );
					break;
					
			}

		} else if ( currDS->dataSetNumber == 0 ) {

			XMP_Uns16 version = GetUns16BE ( valuePtr );
			printf ( ", version = 0x%.4X\n", version );

		} else {
		
			int ds;
			for ( ds = 0; kDataSetNames[ds].name != 0; ++ds ) {
				if ( currDS->dataSetNumber == kDataSetNames[ds].id ) break;
			}
			
			if ( kDataSetNames[ds].name == 0 ) {
				printf ( "\n" );
			} else {
				printf ( ", %s\n", kDataSetNames[ds].name );
				PrintOnlyASCII_8 ( valuePtr, valueLen, "     " );
			}

		}
		
		iptcPtr = valuePtr + valueLen;

	}
	
	printf ( "\n" );
	if ( iptcPtr > iptcEnd ) {
		printf ( "** Too much IPTC data, delta %d\n\n", (long)(iptcEnd - iptcPtr) );
	} else {
		while ( (iptcPtr < iptcEnd) && (*iptcPtr == 0) ) ++iptcPtr;
		if ( iptcPtr != iptcEnd ) printf ( "** Too little IPTC data, delta %d\n\n", (long)(iptcPtr - iptcEnd) );
	}
	
}	// DumpIPTC

// =================================================================================================

#define k8BIM 0x3842494DUL

static void
DumpImageResources ( XMP_Uns8 * psirOrigin, XMP_Uns32 psirLen, XMP_Uns32 fileOffset, const char * label )
{
	printf ( "Photoshop Image Resources from %s, file offset 0x%X, size %d\n\n",
			  label, fileOffset, psirLen );

	XMP_Uns8 * psirPtr = psirOrigin;
	XMP_Uns8 * psirEnd = psirPtr + psirLen;
	XMP_Uns8 * irPtr;
	XMP_Uns32  irType, irLen, irOffset;
	XMP_Uns16  irID;
	const char * irName;

	XMP_Uns8 * iptcPtr = 0;
	XMP_Uns8 * xmpPtr = 0;
	XMP_Uns8 * exif1Ptr = 0;
	XMP_Uns8 * exif3Ptr = 0;
	XMP_Uns32  iptcLen, xmpLen, exif1Len, exif3Len;

	while ( psirPtr < psirEnd ) {

		irType = GetUns32BE ( psirPtr );	// The image resource type, generally "8BIM".
		irID = GetUns16BE ( psirPtr+4 );	// The image resource ID.
		irName = (XMP_StringPtr)psirPtr+6;	// A Pascal string.
		irOffset = 6 + ((*irName + 2) & 0xFFFFFFFE);	// Offset to the image resource data length.
		irLen = GetUns32BE ( psirPtr+irOffset );		// Length of the image resource data.
		irPtr = psirPtr + irOffset + 4;

		irOffset = fileOffset + (psirPtr - psirOrigin);
		printf ( "   '%.4s', #%d, file offset 0x%X, size %d", psirPtr, irID, irOffset, irLen );
		if ( *irName != 0 ) printf ( ", \"%.*s\"", (int)(*irName), (irName+1) );
		
		if ( irType != k8BIM ) {
			printf ( "\n" );
		} else if ( irID == kPSIR_IPTC ) {
			printf ( ", IPTC\n" );
			iptcPtr = irPtr;
			iptcLen = irLen;
		} else if ( irID == kPSIR_XMP ) {
			printf ( ", XMP\n" );
			xmpPtr = irPtr;
			xmpLen = irLen;
		} else if ( irID == kPSIR_Exif_1 ) {
			printf ( ", Exif-1\n" );
			exif1Ptr = irPtr;
			exif1Len = irLen;
		} else if ( irID == kPSIR_Exif_3 ) {
			printf ( ", Exif-3\n" );
			exif3Ptr = irPtr;
			exif3Len = irLen;
		} else if ( irID == kPSIR_IPTC_Digest ) {
			if ( irLen == sizeof(MD5_Digest) ) {
				sHaveOldDigest = true;
				memcpy ( &sOldDigest, irPtr, sizeof(MD5_Digest) );
			}
			printf ( ", IPTC digest = %.8X-%.8X-%.8X-%.8X\n",
					  GetUns32BE(irPtr), GetUns32BE(irPtr+4), GetUns32BE(irPtr+8), GetUns32BE(irPtr+12) );
		} else if ( irID == kPSIR_CopyrightFlag ) {
			bool copyrighted = (*irPtr != 0);
			printf ( ", copyrighted = %s\n", (copyrighted ? "yes" : "no") );
		} else if ( irID == kPSIR_CopyrightURL ) {
			printf ( ", copyright URL\n" );
			PrintOnlyASCII_8 ( irPtr, irLen, "     " );
		} else if ( irID == kPSIR_OldCaption ) {
			printf ( ", old caption\n" );
			PrintOnlyASCII_8 ( irPtr, irLen, "     " );
		} else if ( irID == kPSIR_PrintCaption ) {
			printf ( ", ** obsolete print caption **\n" );
		} else {
			printf ( "\n" );
		}

		psirPtr = irPtr + ((irLen + 1) & 0xFFFFFFFE);	// Round the length to be even.

	}

	printf ( "\n" );
	if ( psirPtr != psirEnd ) {
		printf ( "** Unexpected end of image resources, delta %d\n\n", (long)(psirPtr - psirEnd) );
	}
	
	if ( exif1Ptr != 0 ) DumpTIFF ( exif1Ptr, exif1Len, (fileOffset + (exif1Ptr - psirOrigin)), "PSIR #1058 (Exif 1)" );
	if ( exif3Ptr != 0 ) DumpTIFF ( exif3Ptr, exif3Len, (fileOffset + (exif3Ptr - psirOrigin)), "PSIR #1059 (Exif 3)" );
	if ( iptcPtr != 0 ) DumpIPTC ( iptcPtr, iptcLen, (fileOffset + (iptcPtr - psirOrigin)), "PSIR #1028" );
	if ( xmpPtr != 0 ) DumpXMP ( xmpPtr, xmpLen, (fileOffset + (xmpPtr - psirOrigin)), "PSIR #1060" );

	}	// DumpImageResources

// =================================================================================================

static void
DumpOneIFD ( int ifdIndex, XMP_Uns8 * ifdPtr, XMP_Uns8 * endPtr,
			 XMP_Uns8 * tiffContent, XMP_Uns32 fileOffset, const char * label )
{
	XMP_Uns8 * exifPtr = 0;
	XMP_Uns8 * gpsPtr = 0;
	XMP_Uns8 * interopPtr = 0;
	XMP_Uns8 * makerNotePtr = 0;
	XMP_Uns8 * psirPtr = 0;
	XMP_Uns8 * iptcPtr = 0;
	XMP_Uns8 * xmpPtr = 0;
	XMP_Uns32 psirLen, iptcLen, xmpLen;
	
	XMP_Uns32 ifdOffset = ifdPtr - tiffContent;
	XMP_Uns16 fieldCount = TIFF_GetUns16 ( ifdPtr );
	XMP_Uns32 ifdLen = 2 + (12 * fieldCount) + 4;
	XMP_Uns32 nextIFD = TIFF_GetUns32 ( ifdPtr+ifdLen-4 );

	printf ( "TIFF IFD #%d from %s, IFD offset 0x%X (file 0x%X), tag count %d",
			  ifdIndex, label, ifdOffset, (ifdOffset + fileOffset), fieldCount );

	if ( nextIFD == 0 ) {
		printf ( ", end of IFD chain\n\n" );
	} else {
		printf ( ", next IFD offset 0x%X (file 0x%X)\n\n", nextIFD, (nextIFD + fileOffset) );
	}
	
	XMP_Uns16 prevTag = 0;
	XMP_Uns8 * fieldPtr = tiffContent+ifdOffset+2;
	for ( int i = 0; i < fieldCount; ++i, fieldPtr += 12 ) {

		XMP_Uns16 fieldTag = TIFF_GetUns16 ( fieldPtr );
		XMP_Uns16 fieldType = TIFF_GetUns16 ( fieldPtr+2 );
		XMP_Uns32 valueCount = TIFF_GetUns32 ( fieldPtr+4 );
		XMP_Uns32 valueOffset = TIFF_GetUns32 ( fieldPtr+8 );

		XMP_Uns8 * valuePtr = ifdPtr - ifdOffset + valueOffset;
		XMP_Uns32 valueLen = 0;
		if ( fieldType < kTIFF_TypeEnd ) valueLen = valueCount * sTIFF_TypeSizes[fieldType];
		if ( valueLen <= 4 ) valuePtr = fieldPtr + 8;

		printf ( "   #%d", fieldTag );
		if ( (fieldType < 1) || (fieldType >=  kTIFF_TypeEnd) ) {
			printf ( ", type %d", fieldType );
		} else {
			printf ( ", %s", sTIFF_TypeNames[fieldType] );
		}
		printf ( ", count %d, value size %d", valueCount, valueLen );
		if ( valueLen > 4 ) {
			printf ( ", value offset 0x%X (file 0x%X)", valueOffset, (valueOffset + fileOffset) );
		} else {
			XMP_Uns32 rawValue = GetUns32BE ( fieldPtr+8 );
			printf ( ", value in IFD (0x%.8X)", rawValue );
		}
		
		if ( fieldTag == kTIFF_Exif ) {
			printf ( ", Exif IFD offset" );
			exifPtr = tiffContent + TIFF_GetUns32 ( valuePtr );	// Value is Exif IFD offset.
		} else if ( fieldTag == kTIFF_GPS ) {
			printf ( ", GPS IFD offset" );
			gpsPtr = tiffContent + TIFF_GetUns32 ( valuePtr );	// Value is GPS IFD offset.
		} else if ( fieldTag == kTIFF_Interop ) {
			printf ( ", Interoperability IFD offset" );
			interopPtr = tiffContent + TIFF_GetUns32 ( valuePtr );	// Value is Interoperability IFD offset.
		} else if ( fieldTag == kTIFF_MakerNote ) {	// Decide if the Maker Note might be formatted as an IFD.
			printf ( ", Maker Note" );
			XMP_Uns32 itemCount = (valueLen - 6) / 12;
			if ( (valueLen >= 18) && (valueLen == (6 + itemCount*12)) &&
			     (itemCount == TIFF_GetUns16 ( valuePtr )) &&
			     (TIFF_GetUns32 ( valuePtr+2+(12*itemCount) ) == 0) ) {
				makerNotePtr = valuePtr;
			}
		} else if ( fieldTag == kTIFF_PSIR ) {
			printf ( ", PSIR" );
			psirPtr = valuePtr;
			psirLen = valueLen;
		} else if ( fieldTag == kTIFF_IPTC ) {
			printf ( ", IPTC" );
			iptcPtr = valuePtr;
			iptcLen = valueLen;
		} else if ( fieldTag == kTIFF_XMP ) {
			printf ( ", XMP" );
			if ( fieldType == kTIFF_ASCII ) fieldType = kTIFF_Uns8;	// Avoid displaying the raw packet for mis-typed XMP.
			xmpPtr = valuePtr;
			xmpLen = valueLen;
		} else {
			for ( int i = 0; sTIFF_TagNames[i].tag != 0; ++i ) {
				if ( sTIFF_TagNames[i].tag == fieldTag ) {
					printf ( ", %s", sTIFF_TagNames[i].name );
					break;
				}
			}
		}

		XMP_Uns8 value8;
		XMP_Uns16 value16;
		XMP_Uns32 value32;
		std::string tempStr;
		
		switch ( fieldType ) {
			case kTIFF_Uns8 :
				if ( valueCount == 1 ) {
					value8 = *valuePtr;
					printf ( ", value = %u (0x%.2X)", value8, value8 );
				}
				printf ( "\n" );
				break;
			case kTIFF_ASCII :
				printf ( "\n" );
				PrintOnlyASCII_8 ( valuePtr, valueLen, "     ", false /* ! stopOnNUL */ );
				break;
			case kTIFF_Uns16 :
				if ( valueCount == 1 ) {
					value16 = TIFF_GetUns16 ( valuePtr );
					printf ( ", value = %u (0x%.4X)", value16, value16 );
				}
				printf ( "\n" );
				break;
			case kTIFF_Uns32 :
				if ( valueCount == 1 ) {
					value32 = TIFF_GetUns32 ( valuePtr );
					printf ( ", value = %u (0x%.8X)", value32, value32 );
				}
				printf ( "\n" );
				break;
			case kTIFF_URational :
				if ( valueCount == 1 ) {
					value32 = TIFF_GetUns32 ( valuePtr );
					XMP_Uns32 denom = TIFF_GetUns32 ( valuePtr+4 );
					printf ( ", value = %u/%u", value32, denom );
				}
				printf ( "\n" );
				break;
			case kTIFF_Int8 :
				if ( valueCount == 1 ) {
					value8 = *valuePtr;
					printf ( ", value = %d (0x%.2X)", *((XMP_Int8*)&value8), value8 );
				}
				printf ( "\n" );
				break;
			case kTIFF_Undef8 :
				if ( valueCount == 1 ) {
					value8 = *valuePtr;
					printf ( ", value = 0x%.2X", value8 );
				} else if ( fieldTag == 36864 ) {	// ExifVersion
					printf ( ", value = \"%.*s\"", valueCount, valuePtr );
				} else if ( fieldTag == 37510 ) {	// UserComment
					XMP_Uns8 * encPtr = valuePtr;
					valuePtr += 8;
					valueCount -= 8;
					printf ( "\n      encoding = \"%.8s\"", encPtr  );
					if ( ! CheckBytes ( encPtr, "UNICODE\0", 8 ) ) {
						PrintOnlyASCII_8 ( valuePtr, valueCount, ", value =",
										   false /* ! stopOnNUL */, false /* ! includeNewLine */ );
					} else {
						bool doBE = beTIFF;
						if ( CheckBytes ( valuePtr, "\xFE\xFF", 2 ) ) {
							doBE = true;
							valuePtr += 2;
							valueCount -= 2;
							printf ( ", BE BOM" );
						}
						if ( CheckBytes ( valuePtr, "\xFF\xFE", 2 ) ) {
							doBE = false;
							valuePtr += 2;
							valueCount -= 2;
							printf ( ", LE BOM" );
						}
						if ( doBE ) {
							PrintOnlyASCII_16BE ( (XMP_Uns16*)valuePtr, valueCount, ", value =",
												  false /* ! stopOnNUL */, false /* ! includeNewLine */ );
						} else {
							PrintOnlyASCII_16LE ( (XMP_Uns16*)valuePtr, valueCount, ", value =",
												  false /* ! stopOnNUL */, false /* ! includeNewLine */ );
						}
					}
				}
				printf ( "\n" );
				break;
			case kTIFF_Int16 :
				if ( valueCount == 1 ) {
					value16 = TIFF_GetUns16 ( valuePtr );
					printf ( ", value = %d (0x%.4X)", *((XMP_Int16*)&value16), value16 );
				}
				printf ( "\n" );
				break;
			case kTIFF_Int32 :
				if ( valueCount == 1 ) {
					value32 = TIFF_GetUns32 ( valuePtr );
					printf ( ", value = %d (0x%.8X)", *((XMP_Int32*)&value32), value32 );
				}
				printf ( "\n" );
				break;
			case kTIFF_SRational :
				if ( valueCount == 1 ) {
					value32 = TIFF_GetUns32 ( valuePtr );
					XMP_Uns32 denom = TIFF_GetUns32 ( valuePtr+4 );
					printf ( ", value = %d/%d",
							  *((XMP_Int32*)&value32), *((XMP_Int32*)&denom) );
				}
				printf ( "\n" );
				break;
			case kTIFF_Float :
				break;
			case kTIFF_Double :
				break;
			default :
				printf ( ", ** unknown type **\n" );
				break;
		}
		
		if ( i > 0 ) {
			if ( fieldTag == prevTag ) {
				printf ( "      ** Repeated tag **\n" );
			} else if ( fieldTag < prevTag ) {
				printf ( "      ** Out of order tag **\n" );
			}
		}
		
		prevTag = fieldTag;

	}
	printf ( "\n" );
	
	if ( exifPtr != 0 ) {
		DumpIFDChain ( exifPtr, endPtr, tiffContent, fileOffset, "TIFF tag #34665 (Exif IFD)" );
	}

	if ( gpsPtr != 0 ) {
		DumpIFDChain ( gpsPtr, endPtr, tiffContent, fileOffset, "TIFF tag #34853 (GPS Info IFD)" );
	}

	if ( interopPtr != 0 ) {
		DumpIFDChain ( interopPtr, endPtr, tiffContent, fileOffset, "TIFF tag #40965 (Interoperability IFD)" );
	}

	if ( makerNotePtr != 0 ) {
		DumpIFDChain ( makerNotePtr, endPtr, tiffContent, fileOffset, "TIFF tag #37500 (Maker Note)" );
	}

	if ( psirPtr != 0 ) {
		DumpImageResources ( psirPtr, psirLen, (fileOffset + (psirPtr - tiffContent)), "TIFF tag #34377" );
	}

	if ( iptcPtr != 0 ) {
		DumpIPTC ( iptcPtr, iptcLen, (fileOffset + (iptcPtr - tiffContent)), "TIFF tag #33723" );
	}

	if ( xmpPtr != 0 ) {
		DumpXMP ( xmpPtr, xmpLen, (fileOffset + (xmpPtr - tiffContent)), "TIFF tag #700" );
	}

}	// DumpOneIFD

// =================================================================================================

	
static void
DumpIFDChain ( XMP_Uns8 * startPtr, XMP_Uns8 * endPtr,
			   XMP_Uns8 * tiffContent, XMP_Uns32 fileOrigin, const char * label )
{
	XMP_Uns8 * ifdPtr = startPtr;
	XMP_Uns32  ifdOffset = startPtr - tiffContent;

	for ( size_t ifdIndex = 0; ifdOffset != 0; ++ifdIndex ) {

		if ( (ifdPtr < tiffContent) || (ifdPtr >= endPtr) ) {
			ifdOffset = fileOrigin + (ifdPtr - tiffContent);
			printf ( "** Invalid IFD offset, %d (0x%X) **\n\n", ifdOffset, ifdOffset );
			return;
		}

		XMP_Uns16 fieldCount = TIFF_GetUns16 ( ifdPtr );
		DumpOneIFD ( ifdIndex, ifdPtr, endPtr, tiffContent, fileOrigin, label );
		ifdOffset = TIFF_GetUns32 ( ifdPtr+2+(12*fieldCount) );
		ifdPtr = tiffContent + ifdOffset;

	}

}	// DumpIFDChain

// =================================================================================================

static void
DumpTIFF ( XMP_Uns8 * tiffContent, XMP_Uns32 tiffLen, XMP_Uns32 fileOffset, const char * label )
{
	// ! TIFF can be nested because of the Photoshop 6 wierdness. Save and restore the procs.
	GetUns16_Proc save_GetUns16 = TIFF_GetUns16;
	GetUns32_Proc save_GetUns32 = TIFF_GetUns32;
	GetUns64_Proc save_GetUns64 = TIFF_GetUns64;
	
	if ( CheckBytes(tiffContent,"II\x2A\x00",4) ) {
		beTIFF = false;
		TIFF_GetUns16 = GetUns16LE;
		TIFF_GetUns32 = GetUns32LE;
		TIFF_GetUns64 = GetUns64LE;
		printf ( "Little endian " );
	} else if ( CheckBytes(tiffContent,"MM\x00\x2A",4) ) {
		beTIFF = true;
		TIFF_GetUns16 = GetUns16BE;
		TIFF_GetUns32 = GetUns32BE;
		TIFF_GetUns64 = GetUns64BE;
		printf ( "Big endian " );
	} else {
		printf ( "** Missing TIFF image file header **\n\n" );
		return;
	}

	printf ( "TIFF from %s, file offset 0x%X, size %d\n\n", label, fileOffset, tiffLen );
	
	XMP_Uns32 ifdOffset = TIFF_GetUns32 ( tiffContent+4 );
	DumpIFDChain ( tiffContent+ifdOffset, tiffContent+tiffLen, tiffContent, fileOffset, label );
	
	TIFF_GetUns16 = save_GetUns16;
	TIFF_GetUns32 = save_GetUns32;
	TIFF_GetUns64 = save_GetUns64;

}	// DumpTIFF

// =================================================================================================

static void
DumpPhotoshop ( XMP_Uns8 * psdContent, XMP_Uns32 psdLen )
{
	psdLen=psdLen;	// Avoid unused parameter warning.

	XMP_Uns32  psirOffset = 26 + 4 + GetUns32BE ( psdContent+26 );
	XMP_Uns8 * psirSect = psdContent + psirOffset;
	XMP_Uns8 * psirPtr  = psirSect + 4;
	XMP_Uns32  psirLen = GetUns32BE ( psirSect );

	DumpImageResources ( psirPtr, psirLen, (psirPtr - psdContent), "Photoshop file" );

}	// DumpPhotoshop

// =================================================================================================

static void
DumpJPEG ( XMP_Uns8 * jpegContent, XMP_Uns32 jpegLen )
{
	XMP_Uns8 * endPtr = jpegContent + jpegLen;
	XMP_Uns8 * segPtr = jpegContent;
	XMP_Uns32  segOffset;

	XMP_Uns8 * xmpPtr = 0;
	XMP_Uns8 * psirPtr = 0;
	XMP_Uns8 * exifPtr = 0;
	XMP_Uns16  xmpLen, psirLen, exifLen;
	
	while ( segPtr < endPtr ) {
	
		XMP_Uns16  segMark = GetUns16BE ( segPtr );
		if ( segMark == 0xFFFF ) {
			segPtr += 1;	// Skip leading 0xFF pad byte.
			continue;
		}
		
		XMP_Uns16 minorKind = segMark & 0x000F;
		segOffset = segPtr - jpegContent;
		
		printf ( "   %.4X, file offset 0x%X", segMark, segOffset );
		if ( ((segMark >> 8) != 0xFF) || (segMark == 0xFF00) ) {
			printf ( ", ** invalid JPEG marker **\n" );
			break;
		}
		
		// Check for standalone markers first, only fetch the length for marker segments.

		if ( segMark == 0xFF01 ) {
			printf ( ", ** TEM **\n" );
			segPtr += 2;	// A standalone marker.
			continue;
		} else if ( (0xFFD0 <= segMark) && (segMark <= 0xFFD7) ) {
			printf ( ", RST%d  ** unexpected **\n", minorKind );
			segPtr += 2;	// A standalone marker.
			continue;
		} else if ( segMark == 0xFFD8 ) {
			printf ( ", SOI\n" );
			segPtr += 2;	// A standalone marker.
			continue;
		} else if ( segMark == 0xFFD9 ) {
			printf ( ", EOI\n" );
			segPtr += 2;	// A standalone marker.
			break;	// Exit on EOI.
		}

		XMP_Uns16 segLen = GetUns16BE ( segPtr+2 );

		if ( (0xFFE0 <= segMark) && (segMark <= 0xFFEF) ) {
			const char* segName = (const char *)(segPtr+4);
			printf ( ", size %d, APP%d, \"%s\"", segLen, minorKind, segName );
			if ( (minorKind == 1) &&
				 ((memcmp(segName,"Exif\0\0",6) == 0) || (memcmp(segName,"Exif\0\xFF",6) == 0)) ) {
				printf ( ", Exif" );
				exifPtr = segPtr + 4 + 6;
				exifLen = segLen - 2 - 6;
			} else if ( (minorKind == 13) && (strcmp(segName,"Photoshop 3.0") == 0) ) {
				printf ( ", PSIR" );
				psirPtr = segPtr + 4 + strlen(segName) + 1;
				psirLen = (XMP_Uns16)(segLen - 2 - strlen(segName) - 1);
			} else if ( (minorKind == 1) && (strcmp(segName,"http://ns.adobe.com/xap/1.0/") == 0) ) {
				printf ( ", XMP" );
				xmpPtr = segPtr + 4 + strlen(segName) + 1;
				xmpLen = (XMP_Uns16)(segLen - 2 - strlen(segName) - 1);
			}
			printf ( "\n" );
			segPtr += 2+segLen;
			continue;
		}

		if ( segMark == 0xFFDA ) {
			printf ( ", size %d, SOS", segLen );
			segPtr += 2+segLen;	// Skip the SOS marker segment itself
			long rstCount = 0;
			while ( segPtr < endPtr ) {	// Skip the entropy-coded data and RSTn markers.
				if ( *segPtr != 0xFF ) {
					segPtr += 1;	// General data byte.
				} else {
					segMark = GetUns16BE ( segPtr );
					if ( segMark == 0xFF00 ) {
						segPtr += 2;	// Padded 0xFF data byte.
					} else if ( (segMark < 0xFFD0) || (segMark > 0xFFD7) ) {
						segLen = 0;
						segPtr -= 2;	// Prepare for the increment in the outer loop.
						break;	// Exit, non-RSTn marker.
					} else {
						++rstCount;
						segPtr += 2;
					}
				}
			}
			printf ( ", %d restart markers\n", rstCount );
			segPtr += 2+segLen;
			continue;
		}

		if ( (0xFF02 <= segMark) && (segMark <= 0xFFBF) ) {
			printf ( ", size %d, ** RES **\n", segLen );
		} else if ( (0xFFC0 <= segMark) && (segMark <= 0xFFC3) ) {
			printf ( ", size %d, SOF%d\n", segLen, minorKind );
		} else if ( segMark == 0xFFC4 ) {
			printf ( ", size %d, DHT\n", segLen );
		} else if ( (0xFFC5 <= segMark) && (segMark <= 0xFFC7) ) {
			printf ( ", size %d, SOF%d\n", segLen, minorKind );
		} else if ( segMark == 0xFFC8 ) {
			printf ( ", size %d, JPG\n", segLen );
		} else if ( (0xFFC9 <= segMark) && (segMark <= 0xFFCB) ) {
			printf ( ", size %d, SOF%d\n", segLen, minorKind );
		} else if ( segMark == 0xFFCC ) {
			printf ( ", size %d, DAC\n", segLen );
		} else if ( (0xFFCD <= segMark) && (segMark <= 0xFFCF) ) {
			printf ( ", size %d, SOF%d\n", segLen, minorKind );
		} else if ( segMark == 0xFFDB ) {
			printf ( ", size %d, DQT\n", segLen );
		} else if ( segMark == 0xFFDC ) {
			printf ( ", size %d, DNL\n", segLen );
		} else if ( segMark == 0xFFDD ) {
			printf ( ", size %d, DRI\n", segLen );
		} else if ( segMark == 0xFFDE ) {
			printf ( ", size %d, DHP\n", segLen );
		} else if ( segMark == 0xFFDF ) {
			printf ( ", size %d, EXP\n", segLen );
		} else if ( (0xFFF0 <= segMark) && (segMark <= 0xFFFD) ) {
			printf ( ", size %d, JPG%d\n", segLen, minorKind );
		} else if ( segMark == 0xFFFE ) {
			printf ( ", size %d, COM\n", segLen );
		} else {
			printf ( ", ** UNRECOGNIZED MARKER **\n" );
		}

		segPtr += 2+segLen;
	
	}

	printf ( "\n" );
	if ( segPtr != endPtr ) {
		segOffset = segPtr - jpegContent;
		printf ( "** Unexpected end of JPEG markers at offset %d (0x%X), delta %d **\n\n",
		          segOffset, segOffset, (long)(endPtr-segPtr) );
	}
	
	if ( exifPtr != 0 ) DumpTIFF ( exifPtr, exifLen, (exifPtr - jpegContent), "JPEG Exif APP1" );
	if ( psirPtr != 0 ) DumpImageResources ( psirPtr, psirLen, (psirPtr - jpegContent), "JPEG Photoshop APP13" );
	if ( xmpPtr != 0 ) DumpXMP ( xmpPtr, xmpLen, (xmpPtr - jpegContent), "JPEG XMP APP1" );

}	// DumpJPEG

// =================================================================================================

static void
DumpFile ( const char * fileName )
{
	// Read the first 4K of the file into a local buffer and determine the file format.
	// ! We're using ANSI C calls that don't handle files over 2GB.

	for ( int i = 0; i < 100; ++i ) printf ( "#" ); printf ( "\n" ); 

	FILE * fileRef = fopen ( fileName, "rb" );
	if ( fileRef == 0 ) {
		printf ( "## Can't open \"%s\"\n\n", fileName );
		return;
	}

	fseek ( fileRef, 0, SEEK_END );
	long fileLen = ftell ( fileRef );

	XMP_Uns8 first4K [4096];
	fseek ( fileRef, 0, SEEK_SET );
	fread ( first4K, 1, 4096, fileRef );
	fseek ( fileRef, 0, SEEK_SET );

	XMP_FileFormat format = CheckFileFormat ( fileName, first4K, fileLen );
	if ( format == kXMP_UnknownFile ) {
		printf ( "## Unknown format for \"%s\"\n\n", fileName );
		return;
	}
	
	if ( sXMPPtr != 0 ) free ( sXMPPtr );
	sXMPPtr = 0;
	
	sHaveOldDigest = false;
	
	XMP_Uns8 * fileContent = (XMP_Uns8*) malloc ( fileLen );
	fseek ( fileRef, 0, SEEK_SET );
	fread ( fileContent, 1, fileLen, fileRef );
	fclose ( fileRef );
	
	if ( format == kXMP_JPEGFile ) {
		printf ( "Dumping JPEG file \"%s\", size %d (0x%X)\n\n", fileName, fileLen, fileLen );
		DumpJPEG ( fileContent, fileLen );
	} else if (format == kXMP_TIFFFile ) {
		printf ( "Dumping TIFF file \"%s\", size %d (0x%X)\n\n", fileName, fileLen, fileLen );
		DumpTIFF ( fileContent, fileLen, 0, "TIFF file" );
	} else if (format == kXMP_PhotoshopFile ) {
		printf ( "Dumping Photoshop file \"%s\", size %d (0x%X)\n\n", fileName, fileLen, fileLen );
		DumpPhotoshop ( fileContent, fileLen );
	}

	free ( fileContent );
	
}	// DumpFile

// =================================================================================================

extern "C" int
main ( int argc, const char * argv[] )
{

	printf ( "\n" );
	(void) SXMPMeta::Initialize();

	for ( int i = 1; i < argc; ++i ) {
		try {
			DumpFile ( argv[i] );
		} catch ( XMP_Error & excep ) {
			PrintXMPErrorInfo ( excep, "## Caught XMP error" );
		} catch ( ... ) {
			printf ( "\n## Unknown exception\n" );
		}
	}
	
	SXMPMeta::Terminate();
	printf ( "Done.\n" );
	return 0;

}
